﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using BMS.Utils;

namespace BMS.Facade.Service
{
    /// <summary>
    /// Interface used for manipulate icon operations.
    /// </summary>
    public interface IIconOperations
    {
        /// <summary>
        /// Creates the icon detail.
        /// </summary>
        /// <param name="iconDetail">The icon detail.</param>
        /// <returns></returns>        
        int CreateIconDetail(IconDetail iconDetail);

        /// <summary>
        /// Gets the icon detail.
        /// </summary>
        /// <param name="iconDetailId">The icon detail id.</param>
        /// <param name="facility">The facility.</param>
        /// <returns></returns>        
        IconDetail GetIconDetail(int iconDetailId, Facility facility);

        /// <summary>
        /// Updates the icon detail.
        /// </summary>
        /// <param name="iconDetail">The icon detail.</param>       
        void UpdateIconDetail(IconDetail iconDetail);

        /// <summary>
        /// Deletes the icon detail.
        /// </summary>
        /// <param name="iconDetailId">The icon detail id.</param>
        /// <param name="facility">The facility.</param>        
        void DeleteIconDetail(int iconDetailId, Facility facility);

        /// <summary>
        /// Switches the icons.
        /// </summary>
        /// <param name="iconId1">The icon id1.</param>
        /// <param name="iconId2">The icon id2.</param>
        /// <param name="facilityUid">The facility uid.</param>
        /// <param name="group">The group.</param>
        void SwitchIcons(int iconId1, int iconId2, Guid? facilityUid, string group);

        /// <summary>
        /// Updates the icon order for a pre-sorted array of icon ids.
        /// </summary>
        /// <param name="iconIds">The icons ids in order.</param>
        /// <param name="group">The group.</param>
        /// <param name="group">The sub group.</param>
        void UpdateIconOrder(string[] iconIds, string group, string subGroup);

        /// <summary>
        /// Updates the icon order bulk.
        /// </summary>
        /// <param name="icons">The icons.</param>
        /// <param name="facility">The facility.</param>
        List<IconInfo> UpdateIconOrderBulk(List<IconInfo> icons, Facility facility);

        /// <summary>
        /// Gets all global icons(all subgroups, including SiteConfigurable defined in Admin section).
        /// </summary>
        /// <returns></returns>
        List<IconInfo> GetAllGlobalIcons();

        /// <summary>
        /// Gets all icons by facility(all subgroups, including SiteConfigurable defined for the specified facility).
        /// </summary>
        /// <param name="facilityUid">The facility uid.</param>
        /// <returns></returns>
        List<IconInfo> GetAllIconsByFacility(Guid facilityUid);

        /// <summary>
        /// Gets the whiteboard icons(all whiteboard subgroups, including SiteConfigurable defined for the specified facility).
        /// </summary>
        /// <param name="facilityUid">The facility uid.</param>
        /// <returns></returns>
        List<IconInfo> GetWhiteboardIcons(Guid facilityUid);

        /// <summary>
        /// Gets the site configurable icons.
        /// </summary>
        /// <param name="facilityUid">The facility uid.</param>
        /// <returns></returns>
        List<IconInfo> GetSiteConfigurableIcons(Guid facilityUid);

        /// <summary>
        /// Searches the icons.
        /// </summary>
        /// <param name="iconFlagIds">The icon flag ids.</param>
        /// <param name="sortBy">The sort by.</param>
        /// <returns></returns>
        List<IconInfo> SearchIcons(string iconFlagIds);

        /// <summary>
        /// Gets the icon by code.
        /// </summary>
        /// <param name="code">The code.</param>
        /// <returns></returns>
        IconInfo GetIconByCode(string code);

        /// <summary>
        /// Gets the icons by codes.
        /// </summary>
        /// <param name="codes">The codes.</param>
        /// <returns></returns>
        Dictionary<string, IconInfo> GetIconsByCodes(List<string> codes);

        /// <summary>
        /// Filters the bed patient icon association.
        /// </summary>
        /// <param name="bedUid">The bed uid.</param>
        /// <param name="patientUid">The patient uid.</param>
        /// <param name="facilityUid">The facility uid.</param>
        /// <returns></returns>
        Dictionary<string, List<IconAssociationInfo>> FilterBedPatientIconAssociation(Guid bedUid, Guid? patientUid, Guid facilityUid);

        /// <summary>
        /// Filters the bed icon association.
        /// </summary>
        /// <param name="bedUids">The bed uids.</param>
        /// <param name="facilityUID">The facility UID.</param>
        /// <returns></returns>
        List<IconAssociation> FilterBedIconAssociation(string bedUids, Guid facilityUID);

        Dictionary<Guid, PatientIcon> GetPatientIcons(string group, string subGroup, string patientUidList);

        /// <summary>
        /// Inserts the icon association.
        /// </summary>
        /// <param name="iconAssociation">The icon association.</param>
        void InsertIconAssociation(IconAssociation iconAssociation);
    }
}
